<?php
/*
  Plugin Name: Dormatic Core
  Plugin URI: http://smartdatasoft.com/
  Description: Helping for the Dormatic theme.
  Author: SmartDataSoft
  Version: 1.7
  Author URI: http://smartdatasoft.com/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once __DIR__ . '/breadcrumb-navxt/breadcrumb-navxt.php';
require_once __DIR__ . '/vendor/autoload.php';
require_once __DIR__ . '/combine-vc-ele-css/combine-vc-ele-css.php';
require_once __DIR__ . '/page-option/page-option.php';

/**
 * The main plugin class
 */
final class Dormatic_Helper {

	/**
	 * Plugin version
	 *
	 * @var string
	 */
	const version = '1.0';


	/**
	 * Plugin Version
	 *
	 * @since 1.2.0
	 * @var   string The plugin version.
	 */
	const VERSION = '1.2.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.2.0
	 * @var   string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.2.0
	 * @var   string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Constructor
	 *
	 * @since  1.0.0
	 * @access public
	 */

	/**
	 * Class construcotr
	 */
	private function __construct() {
		$this->define_constants();
		add_action( 'plugins_loaded', array( $this, 'init_plugin' ) );
	}

	/**
	 * Initializes a singleton instance
	 *
	 * @return \Dormatic
	 */
	public static function init() {
		static $instance = false;

		if ( ! $instance ) {
			$instance = new self();
		}

		return $instance;
	}


	/**
	 * Define the required plugin constants
	 *
	 * @return void
	 */
	public function define_constants() {
		define( 'Dormatic_CORE_VERSION', self::version );
		define( 'Dormatic_CORE_FILE', __FILE__ );
		define( 'Dormatic_CORE_PATH', __DIR__ );
		define( 'Dormatic_CORE_URL', plugin_dir_url( __FILE__ ) );
		define( 'Dormatic_CORE_ASSETS_DEPENDENCY_CSS', Dormatic_CORE_URL . '/assets/elementor/css/' );
		define( 'Dormatic_CORE_ASSETS', Dormatic_CORE_URL . 'assets' );
		$theme = wp_get_theme();
		define( 'THEME_VERSION_CORE', $theme->Version );
	}

	/**
	 * Initialize the plugin
	 *
	 * @return void
	 */
	public function init_plugin() {
		$this->checkElementor();
		load_plugin_textdomain( 'dormatic-core', false, basename( dirname( __FILE__ ) ) . '/languages' );
		new Dormatic\Helper\Posttype();
		
		// sidebar generator
		new \Dormatic\Helper\Sidebar_Generator();

		new \Dormatic\Helper\Widgets();
		if ( did_action( 'elementor/loaded' ) ) {
			new \Dormatic\Helper\Elementor();
		}

		if ( is_admin() ) {
			new \Dormatic\Helper\Admin();
		}
	}

	public function checkElementor() {
		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_elementor_version' ) );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_php_version' ) );
			return;
		}
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}
		$message = '<p>If you want to use Elementor Version of "<strong>dormatic</strong>" Theme, Its requires "<strong>Elementor</strong>" to be installed and activated.</p>';
		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
		/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'elementor-hello-world' ),
			'<strong>' . esc_html__( 'Elementor ChildIt', 'elementor-hello-world' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'elementor-hello-world' ) . '</strong>',
			self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
		/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'dormatic-core' ),
			'<strong>' . esc_html__( 'Elementor ChildIt', 'dormatic-core' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'dormatic-core' ) . '</strong>',
			self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

}

/**
 * Initializes the main plugin
 *
 * @return \Dormatic_Init
 */
function Dormatic_Init() {
	return Dormatic_Helper::init();
}

// kick-off the plugin
Dormatic_Init();


// Get The Menu List
function dormatic_get_menu_list() {
	$menulist = array();
	$menus    = get_terms( 'nav_menu' );
	foreach ( $menus as $menu ) {
		$menulist[ $menu->name ] = $menu->name;
	}
	return $menulist;
}


// Enqueue Style During Editing
add_action(
	'elementor/editor/before_enqueue_styles',
	function () {
		wp_enqueue_style( 'elementor-stylesheet', plugins_url() . '/dormatic-core/assets/elementor/stylesheets.css', true, time() );
	}
);

/**
 * Passing Classes to Menu
 */
add_action(
	'wp_nav_menu_item_custom_fields',
	function( $item_id, $item ) {
		if ( $item->menu_item_parent == '0' ) {
			$show_as_megamenu = get_post_meta( $item_id, '_show-as-megamenu', true ); ?>
			<p class="description-wide">
				<label for="megamenu-item-<?php echo $item_id; ?>"> <input type="checkbox"
						id="megamenu-item-<?php echo $item_id; ?>" name="megamenu-item[<?php echo $item_id; ?>]"
						<?php checked( $show_as_megamenu, true ); ?> /><?php _e( 'Mega menu', 'sds' ); ?>
				</label>
			</p>
			<?php
		}
	},
	10,
	2
);

add_action(
	'wp_update_nav_menu_item',
	function( $menu_id, $menu_item_db_id ) {
		$button_value = ( isset( $_POST['megamenu-item'][ $menu_item_db_id ] ) && $_POST['megamenu-item'][ $menu_item_db_id ] == 'on' ) ? true : false;
		update_post_meta( $menu_item_db_id, '_show-as-megamenu', $button_value );
	},
	10,
	2
);

add_filter(
	'nav_menu_css_class',
	function( $classes, $menu_item ) {
		if ( $menu_item->menu_item_parent == '0' ) {
			$show_as_megamenu = get_post_meta( $menu_item->ID, '_show-as-megamenu', true );
			if ( $show_as_megamenu ) {
				$classes[] = 'megamenu';
			}
		}
		return $classes;
	},
	10,
	2
);

// Hook called when the plugin is activated.
add_action( 'plugins_loaded', 'dormatic_function_elem_cpt_support' );
function dormatic_function_elem_cpt_support() {
	$cpt_support = get_option( 'elementor_cpt_support' );
	if ( ! $cpt_support ) {
		// First check if the option is not available already in the database. It not then create the array with all default post types including yours and save the settings.
		$cpt_support = array( 'page', 'post', 'service' );
		update_option( 'elementor_cpt_support', $cpt_support );
	} elseif ( ! in_array( 'service', $cpt_support ) ) {
		// If the option is available then just append the array and update the settings.
		$cpt_support[] = 'service';
		update_option( 'elementor_cpt_support', $cpt_support );
	}
}

// add support link to the WP Toolbar
function dormatic_core_toolbar_link($wp_admin_bar) {
    $args = array(
        'id' => 'support_link',
        'title' => esc_html__('Theme Support & Documentation links','dormatic-core'), 
        'href' => home_url() . '/wp-admin/admin.php?page=envato-theme-license-support', 
        'meta' => array(
            'class' => 'sp_link', 
            'title' => esc_html__('Support','dormatic-core')
            )
    );
    $wp_admin_bar->add_node($args);
}
add_action('admin_bar_menu', 'dormatic_core_toolbar_link', 999);